' NOMIS, a Simon Clone game for CMM2
' Rev 1.0.0 William M Leue 12-August-2023
' this version requires a mouse

option default integer
option base 1
option angle degrees

' Constants

' general game params
const NBUTTONS = 4
const MAX_BEEPS = 20
const CASE_COLOR = rgb(black)
const NUM_WHITE_NOTES = 22
const NUM_BLACK_NOTES = 16
const USER = 1
const COMPUTER = 2

' display params
const CASE_RAD         = 250
const BUTTON_MARGIN    = 20
const BUTTON_INNER_RAD = 80
const BUTTON_OUTER_RAD = CASE_RAD - BUTTON_MARGIN
const BUTTON_HVGAP     = int((1.0*CASE_RAD)/(20.0) + 0.5)
const BUTTON_NRADPTS   = 20
const BUTTON_NVERTS    = 4+2*BUTTON_NRADPTS

' keyboard codes
const UP    = 128
const DOWN  = 129
const LEFT  = 130
const RIGHT = 131
const ENTER = 13
const ESC   = 27

' tunes and flourish params
const NWNOTES   = 6
const NDNOTES   = 8
const WIN_TUNE  = 1
const LOSE_TUNE = 2
const NFLOURISH = 12
const F_LENGTH  = 100

' Give the user a small break
const UTIMEADD  = 0.30

' Controls
const HLPX  = 10
const CTBY  = 5
const CTBW  = 80
const CTBG  = 20
const CTBH  = 30
const HLPH  = CTBH
const HLPR  = CTBH\2
const OPTX  = HLPX+CTBW+CTBG
const OPTR  = CTBH\2
const PLAX  = HLPX+2*CTBW+2*CTBG
const QITX  = HLPX+3*CTBW+3*CTBG
const CTBTW = 4*CTBW + 3*CTBG
const CTBC  = rgb(0, 128, 0)
const CTBC_FLASH = rgb(green)

' Options
const OPT_NBEEPS = 1
const OPT_SPEED  = 2
const OPT_FONLY  = 3
const OPT_TONLY  = 4
const OPT_BACKW  = 5
const NOPTS = 5

' Mouse I2C channel
const MCHAN = 2

' Globals
dim prebeeps(MAX_BEEPS)
dim beeps(MAX_BEEPS)
dim buttons(NBUTTONS)
dim ON_colors(NBUTTONS) =  (rgb(red), rgb(green), rgb(blue), rgb(yellow))
dim OFF_colors(NBUTTONS) = (rgb(96, 0, 0), rgb(0, 96, 0), rgb(0, 0, 96), rgb(96, 96, 0))
dim white_notes(NUM_WHITE_NOTES)
dim black_notes(NUM_BLACK_NOTES)
dim bvertx(NBUTTONS, BUTTON_NVERTS)
dim bverty(NBUTTONS, BUTTON_NVERTS)
dim bnotes(NBUTTONS)
dim wnotes(NWNOTES)
dim dnotes(NDNOTES)
dim windices(NWNOTES) = (12, 13, 14, 16, 14, 16)
dim wlengths(NWNOTES) = (100, 100, 100, 300, 100, 300)
dim dindices(NDNOTES) = (5, 4, 5, 3, 5, 4, 3, 3)
dim dlengths(NDNOTES) = (500, 500, 500, 500, 500, 1000, 200, 600)
dim noptvals(NOPTS) = (3, 3, 2, 2, 2)
dim option_values(NOPTS, 3)
dim option_text$(NOPTS)
dim option_choice(NOPTS)
dim game_beeps = 0
dim game_speed = 0
dim flash_only = 0
dim tones_only = 0
dim backwards  = 0
dim nbeeps = 0
dim beep_length = 0
dim beep_delay = 0
dim cx = mm.hres\2
dim cy = mm.vres\2
dim mouse_x = 0
dim mouse_y = 0
dim lclick = 0
dim timeout = 0
dim user_nbeeps = 0
dim start_play = 0
dim running = 0
dim bccount = 0
dim bcbutton = 0

' Main Program
'open "debug.txt" for output as #1
ReadSounds
MakeButtonPolygons
MakeOptionValues
if MCHAN > 0 then InitMouse
DrawScreen
Flourish
PlayLoop
end

' Read the note frequency values and set up the button tones.
' (The buttons are E, C#, A, and E (one octave below), which
' is an E major chord, 2nd inversion.)
sub ReadSounds
  local i
  for i = 1 to NUM_WHITE_NOTES
    read white_notes(i)
  next i
  for i = 1 to NUM_BLACK_NOTES
    read black_notes(i)
  next i
  bnotes(1) = white_notes(14)
  bnotes(2) = black_notes(9)
  bnotes(3) = white_notes(10)
  bnotes(4) = white_notes(7)
  ' a couple of short, cheerful (?) tunes for winning and losing
  for i = 1 to NWNOTES
    wnotes(i) = white_notes(windices(i))
  next i
  for i = 1 to NDNOTES
    dnotes(i) = white_notes(dindices(i))
  next i
end sub

' Make arrays of option values
sub MakeOptionValues
  local i
  option_text$(OPT_NBEEPS) = "Number of Tones in Game:"
  option_text$(OPT_SPEED) = "Game Speed:"
  option_text$(OPT_FONLY) = "Light Flashes Only:"
  option_text$(OPT_TONLY) = "Tones Only:"
  option_text$(OPT_BACKW) = "Backwards:"
  option_values(OPT_NBEEPS, 1) = 8
  option_values(OPT_NBEEPS, 2) = 14
  option_values(OPT_NBEEPS, 3) = 20
  option_values(OPT_SPEED, 1) = 1
  option_values(OPT_SPEED, 2) = 2
  option_values(OPT_SPEED, 3) = 3
  option_values(OPT_FONLY, 1) = 0
  option_values(OPT_FONLY, 2) = 1
  option_values(OPT_FONLY, 3) = 0
  option_values(OPT_TONLY, 1) = 0
  option_values(OPT_TONLY, 2) = 1
  option_values(OPT_TONLY, 3) = 0
  option_values(OPT_BACKW, 1) = 0
  option_values(OPT_BACKW, 2) = 1
  option_values(OPT_BACKW, 3) = 0
  for i = 1 to NOPTS
    option_choice(i) = 1
  next i
end sub

' Make the vertices for the polygons for the 4 buttons
sub MakeButtonPolygons
  local x1, y1, x2, y2, x3, y3, x4, y4
  local i, j, blen
  local float gangle, sangle, angle, ainc, gratio, rratio, bangle
  blen = BUTTON_OUTER_RAD - BUTTON_INNER_RAD
  gratio = (1.0*BUTTON_INNER_RAD)/(1.0*BUTTON_HVGAP)
  rratio = PI*gratio/180.0
  gangle = asin(rgratio)
  ainc = (90.0-2*gangle)/BUTTON_NRADPTS
  for i = 1 to NBUTTONS
    bangle = (i-1)*360.0/NBUTTONS
    sangle = bangle + gangle
    select case i
      case 1
        x1 = cx + BUTTON_INNER_RAD
        y1 = cy - BUTTON_HVGAP
        x2 = cx + BUTTON_OUTER_RAD
        y2 = cy - BUTTON_HVGAP
        x3 = cx + BUTTON_HVGAP
        y3 = cy - BUTTON_OUTER_RAD
        x4 = cx + BUTTON_HVGAP
        y4 = cy - BUTTON_INNER_RAD
      case 2
        x1 = cx - BUTTON_HVGAP
        y1 = cy - BUTTON_INNER_RAD
        x2 = cx - BUTTON_HVGAP
        y2 = cy - BUTTON_OUTER_RAD
        x3 = cx - BUTTON_OUTER_RAD
        y3 = cy - BUTTON_HVGAP
        x4 = cx - BUTTON_INNER_RAD
        y4 = cy - BUTTON_HVGAP
      case 3
        x1 = cx - BUTTON_INNER_RAD
        y1 = cy + BUTTON_HVGAP
        x2 = cx - BUTTON_OUTER_RAD
        y2 = cy + BUTTON_HVGAP
        x3 = cx - BUTTON_HVGAP
        y3 = cy + BUTTON_OUTER_RAD
        x4 = cx - BUTTON_HVGAP
        y4 = cy + BUTTON_INNER_RAD
      case 4
        x1 = cx + BUTTON_HVGAP
        y1 = cy + BUTTON_INNER_RAD
        x2 = cx + BUTTON_HVGAP
        y2 = cy + BUTTON_OUTER_RAD
        x3 = cx + BUTTON_OUTER_RAD
        y3 = cy + BUTTON_HVGAP
        x4 = cx + BUTTON_INNER_RAD
        y4 = cy + BUTTON_HVGAP
    end select  
    bvertx(i, 1) = x1  :  bverty(i, 1) = y1
    bvertx(i, 2) = x2  :  bverty(i, 2) = y2
    angle = sangle
    for j = 1 to BUTTON_NRADPTS
      bvertx(i, j+2) = cx + int(BUTTON_OUTER_RAD*cos(angle) + 0.5)
      bverty(i, j+2) = cy - int(BUTTON_OUTER_RAD*sin(angle) + 0.5)
      inc angle, ainc
    next j
    bvertx(i, BUTTON_NRADPTS+3) = x3 : bverty(i, BUTTON_NRADPTS+3) = y3
    bvertx(i, BUTTON_NRADPTS+4) = x4 : bverty(i, BUTTON_NRADPTS+4) = y4
    angle = sangle + 360.0/NBUTTONS
    for j = 1 to BUTTON_NRADPTS
      bvertx(i, j+BUTTON_NRADPTS+4) = cx + int(BUTTON_INNER_RAD*cos(angle) + 0.5)
      bverty(i, j+BUTTON_NRADPTS+4) = cy - int(BUTTON_INNER_RAD*sin(angle) + 0.5)
      inc angle, -ainc
    next j
  next i  
end sub

' Initialize the mouse and cursor if we have one
sub InitMouse
  on error  skip 1
  controller mouse open MCHAN, LeftClick
  if mm.errno <> 0 then
    cls
    print "Sorry, this version of NOMIS requires a mouse!"
    end
  end if
  gui cursor on 1
  settick 20, UpdateCursor, 1
end sub

' Mouse Left Click ISR
sub LeftClick
  mouse_x = mouse(X)
  mouse_y = mouse(Y)
  lclick = 1
end sub

' Update cursor to track the mouse
sub UpdateCursor
  gui cursor mouse(X), mouse(Y)
end sub

' Draw the screen
sub DrawScreen
  local i
  cls rgb(gray)
  DrawControls 0
  circle cx, cy, CASE_RAD,,, CASE_COLOR, CASE_COLOR
  for i = 1 to NBUTTONS
    DrawButton i, 0
  next i
end sub

' Draw the Controls
' If which > 0 then flash the corresponding control button brighter
sub DrawControls which
  local c(4) = (CTBC, CTBC, CTBC, CTBC)
  if which > 0 then c(which) = CTBC_FLASH
  gui cursor hide
  rbox HLPX, CTBY, CTBW, CTBH, HLPR, rgb(black), c(1)
  text HLPX+CTBW\2, CTBY+CTBH\2, "Help", "CM",,, rgb(black), -1
  rbox OPTX, CTBY, CTBW, CTBH, OPTR, rgb(black), c(2)
  text OPTX+CTBW\2, CTBY+CTBH\2, "Options", "CM",,, rgb(black), -1
  rbox PLAX, CTBY, CTBW, CTBH, OPTR, rgb(black), c(3)
  text PLAX+CTBW\2, CTBY+CTBH\2, "Play!", "CM",,, rgb(black), -1
  rbox QITX, CTBY, CTBW, CTBH, OPTR, rgb(black), c(4)
  text QITX+CTBW\2, CTBY+CTBH\2, "Quit", "CM",,, rgb(black), -1
  gui cursor show
end sub

' Draw the specified button with dim colors (not active) or bright (active)
sub DrawButton which, active
  local c, mlen, mwid
  local xv(BUTTON_NVERTS), yv(BUTTON_NVERTS)
  gui cursor hide
  if active then
    c = ON_colors(which)
  else
    c = OFF_colors(which)
  end if
  math slice bvertx(), which,, xv()
  math slice bverty(), which,, yv()
  polygon BUTTON_NVERTS, xv(), yv(), c, c
  mlen = 2*BUTTON_OUTER_RAD+1
  mwid = 2*BUTTON_HVGAP
  box cx-BUTTON_OUTER_RAD, cy-BUTTON_HVGAP, mlen, mwid,, CASE_COLOR, CASE_COLOR
  box cx-BUTTON_HVGAP, cy-BUTTON_OUTER_RAD, mwid, mlen,, CASE_COLOR, CASE_COLOR
  text cx, cy, "NOMIS", "CM", 4
  gui cursor show
end sub

' Play the current number of beeps and flash the corresponding buttons
sub PlayBeeps
  local i, note, button
  for i = 1 to nbeeps
    button = beeps(i)
    PlayBeep button, COMPUTER
if i = 3 then save image "NOMIS"
  next i
end sub

' Play a single beep
sub PlayBeep button, who
  local note, flash, toval, foval
  flash = 1
  toval = option_values(OPT_TONLY, option_choice(OPT_TONLY))
  foval = option_values(OPT_FONLY, option_choice(OPT_FONLY))
  if toval then flash = 0
  note = bnotes(button)
  DrawButton button, flash
  if not foval then
    play tone note, note, beep_length
  end if
  pause beep_delay
  DrawButton button, 0
  if who = COMPUTER then pause beep_delay
end sub

' Handle multiple games
sub PlayLoop
  do
    if lclick then
      cb = GetControlButton()
      if cb > 0 then
        DoControlButton cb
      end if
      lclick = 0
    end if
    if start_play then
      start_play = 0
      InitGame
      PlayGame
    end if
  loop
end sub

' Initialize a new game
sub InitGame
  local i
  game_beeps = option_values(OPT_NBEEPS, option_choice(1))
  select case option_choice(OPT_SPEED)
    case 1
      beep_length = 500
      beep_delay =  350
    case 2
      beep_length = 750
      beep_delay  = 500
    case 3
      beep_length = 350
      beep_delay = 250
  end select
  nbeeps = 0
  user_nbeeps = 0
  timeout = 0
  MakePrebeeps
  bccount = 0
  bcbutton = 0
  lclick = 0
  running = 1
end sub

' Play a game
sub PlayGame
  local tb
  local ptimer, utimer, allowed
  do
    if not running then exit do
    inc nbeeps
    beeps(nbeeps) = prebeeps(nbeeps)
    timer = 0
    pause 1000
    PlayBeeps
    ptimer = timer 
    user_nbeeps = 0
    allowed = 1000 + ptimer + int(ptimer*UTIMEADD)
    timer = 0
    HandleUserClicks 
    if running then
      utimer = timer
      if utimer > allowed then
        TooSlow
        exit do
      end if    
    end if
  loop until (nbeeps = game_beeps) or not running
end sub

' Make up a list of pseudo-random buttons
' This is a bit 'cooked' from a strictly uniform random
' distribution to get rid of excessive consecutive beeps
' on the same button.
sub MakePrebeeps
  local i, j, t, r
  for i = 1 to MAX_BEEPS
    beeps(i) = 0
  next i
  for i = 1 to game_beeps
    prebeeps(i) = ((i-1) mod NBUTTONS) + 1
  next i
  for i = 1 to 5
    for j = 1 to game_beeps-1
      r = RandInt(1, game_beeps)
      t = prebeeps(r)
      prebeeps(r) = prebeeps(j)
      prebeeps(j) = t
    next j
  next i
end sub

' Handle a user mouse button clicks until sequence complete
' or a wrong button is pressed.
sub HandleUserClicks
  local button, cb, bkw, bx
  bkw = option_values(OPT_BACKW, option_choice(OPT_BACKW))
  do
    if lclick then
      button = GetClickedButton()
      if button > 0 then
        inc user_nbeeps
        PlayBeep button, USER
        if bkw then
          bx = nbeeps - user_nbeeps + 1
        else
          bx = user_nbeeps
        end if
        if button <> beeps(bx) then
          running = 0
          ShowLostGame "Wrong button -- You lose!"
          exit sub
        else
        end if
        if user_nbeeps = game_beeps then
          ShowWonGame
          exit sub
        end if
      end if
      lclick = 0
    end if
  loop until (user_nbeeps = nbeeps) or not running
end sub

' analyze mouse click and return the button index,
' or zero if not in a button
function GetClickedButton()
  local float radius
  local x, y, b
  GetClickedButton = 0
  x = mouse_x - cx
  y = cy - mouse_y
  radius = sqr(x*x + y*y)
  if radius < BUTTON_INNER_RAD or radius > BUTTON_OUTER_RAD then
    exit function
  end if
  if x > -BUTTON_HVGAP and x < BUTTON_HVGAP then
    exit function
  end if
  if y > -BUTTON_HVGAP and y < BUTTON_HVGAP then
    exit function
  end if
  if x > 0 then
    if y > 0 then
      b = 1
    else
      b = 4
    end if
  else
    if y > 0 then
      b = 2
    else
      b = 3
    end if
  end if
  GetClickedButton = b
end function

' handle timeout on user input
sub TooSlow
  timeout = 1
  running = 0
  pause 100
  ShowLostGame "You Are Too Slow -- You Lose!"
end sub

' show the user losing
sub ShowLostGame m$
  running = 0
  ShowMessage m$, rgb(red), rgb(black)
  PlayTune LOSE_TUNE
  DrawScreen
end sub

' show the user winning
sub ShowWonGame
  pause 100
  running = 0
  ShowMessage "You Win!!", rgb(green), rgb(black)
  PlayTune WIN_TUNE
  DrawScreen
end sub

' Show a text message
sub ShowMessage m$, bc, tc
  local bw = 400
  local bh = 200
  box mm.hres\2 - bw\2, mm.vres\2 - bh\2, bw, bh,, rgb(black), bc
  text mm.hres\2, mm.vres\2, m$, "CM", 4,, tc, -1
end sub

' Play a tune for winning or losing
sub PlayTune which
  local i
  select case which
    case WIN_TUNE
      for i = 1 to NWNOTES
        play tone wnotes(i), wnotes(i), wlengths(i)
        pause wlengths(i) + 50
      next i
    case LOSE_TUNE
      for i = 1 to NDNOTES
        play tone dnotes(i), dnotes(i), dlengths(i)
        pause dlengths(i) + 50
      next i
  end select
end sub

' Start off with a flourish of lights and tones
sub Flourish
  local i, b
  beep_length = 200
  beep_delay = 100
  for i = NFLOURISH to 1 step -1
    b = ((i-1) mod NBUTTONS) + 1
    PlayBeep b, COMPUTER
  next i
end sub

' Return the control button index or zero if not a click
' in a control button
function GetControlButton()
  bx = 0
  GetControlButton = bx
  if mouse_y < CTBY or mouse_y > CTBY+CTBH then exit function
  if mouse_x < HLPX or mouse_x > HLPX + CTBTW then exit function
  if mouse_x >= HLPX and mouse_x <= HLPX+CTBW then bx = 1
  if mouse_x >= OPTX and mouse_x <= OPTX+CTBW then bx = 2
  if mouse_x >= PLAX and mouse_x <= PLAX+CTBW then bx = 3
  if mouse_x >= QITX and mouse_x <= QITX+CTBW then bx = 4
  DrawControls bx
  pause 100
  DrawControls 0
  GetControlButton = bx
end function

' Handle Control Button Clicks
sub DoControlButton which
  running = 0
  select case which
    case 1 : ShowInstructions
    case 2 : ShowOptions
    case 3 : start_play = 1
    case 4 : Quit
  end select
end sub
        
' return a uniformly distributed random integer in the specified closed range
function RandInt(a as integer, b as integer)
  local integer v, c
  c = b-a+1
  do
    v = a + (b-a+2)*rnd()
    if v >= a and v <= b then exit do
  loop
  RandInt = v
end function

' Show Instructions
sub ShowInstructions
  local z$
  cls
  text mm.hres\2, 10, "Instructions for Playing Simon", "CT", 4,, rgb(green)
  print @(0, 40) "NOMIS is a near-clone of the classic Simon game."
  print ""
  print "The 4 large colored buttons play tones when pressed, either by you or by the"
  print "computer. A game starts with the computer playing a tone and flashing the"
  print "corresponding button. You must press that same button to keep playing. Next,
  print "the computer plays and flashes two buttons, starting with the same one as before."
  print "You have to press those same 2 buttons. As the game progresses, the sequence of"
  print "button presses and tones gets longer and longer. If you press the wrong button or"
  print "hesitate too long, the game is over.
  print ""
  print "The normal game stops after 8 tones. If you get them all correctly, you win!"
  print ""
  print "There are lots of options you can choose to make the game more challenging and more"
  print "interesting:"
  print ""
  print "  Number of Tones: (8, 14, 20)         [ Default = 8 ]
  print "  Speed: (normal, slow, fast)          [ Default = normal]
  print "  Suppress tones: (0=No, 1=Yes)        [ Default = No]
  print "  Suppress flashes: (0=No, 1=Yes)      [ Default = No]
  print "  Play tones BACKWARDS: (0=No, 1=Yes)  [ Default = No]
  print ""
  print "(and yes, you can suppress BOTH the tones and the button flashes. I guess this counts"
  print "as the most difficult game of all :-)"
  print ""
  print "Click the Options button at the top of the main screen to change Options.
  text mm.hres\2, 585, "Press Any Key to Continue", "CB"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  cls
  DrawScreen
end sub

' Show and let user choose options
sub ShowOptions
  local z$, cmd, sel
  cls
  text mm.hres\2, 10, "Game Play Options", "CT", 4,, rgb(green)
  print @(0, 40) "Press the Left or Right Arrow key to cycle through the values"
  print "for an option category. Press the Up or Down arrow key to move to different"
  print "categories. Press Enter to accept the current options and return to the game."
  sel = 1
  DrawOptions sel
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(UCASE$(z$))
    select case cmd
      case UP
        if sel > 1 then
          inc sel, -1
        else
          sel = NOPTS
        end if
        DrawOptions sel
      case DOWN
        if sel < NOPTS then
          inc sel
        else
          sel = 1
        end if
        DrawOptions sel
      case LEFT
        if option_choice(sel) > 1 then
          inc option_choice(sel), -1
        else
          option_choice(sel) = noptvals(sel)
        end if
        DrawOptions sel
      case RIGHT
        if option_choice(sel) < noptvals(sel)  then
          inc option_choice(sel), 1
        else
          option_choice(sel) = 1
        end if
        DrawOptions sel
      case ENTER
        exit do
    end select
  loop
  z$ = INKEY$
  DrawScreen
end sub      

' Draw the Options
sub DrawOptions sel
  local i, tc, bc, m$, v
  tc = rgb(white) : bc = rgb(black)
  for i = 1 to NOPTS
    v = option_values(i, option_choice(i))
    m$ = option_text$(i) + " " + str$(v) + " "
    if i = sel then
      text 30, 100+(i-1)*30, m$,,,, bc, tc
    else
      text 30, 100+(i-1)*30, m$,,,, tc, bc
    end if
  next i
end sub

' Quit cleanly
sub Quit
  gui cursor off
  controller mouse close
  cls
  end
end sub

' Note frequencies
' white keys
data 87.307, 98.0, 110.0, 123.74, 130.81, 146.83, 164.81, 174.61, 196.0, 220.0, 246.94, 261.63
data 293.66, 329.63, 349.23, 392.0, 440.0, 493.88, 523.25, 587.33, 659.26, 698.456
' black keys
data 92.499, 103.826, 116.541, 138.591, 155.563, 184.997, 207.652, 233.082, 277.183, 311.127, 369.994
data 415.305, 466.164, 554.183, 622.254, 739.989


